// WEBSOCKET CLIENT PARA ALERTA VES - NOTIFICACIONES PARA TODOS
// Archivo: /frontend/Assets/js/websocket.js

class AlertWebSocketClient {
    constructor() {
        this.connected = false;
        this.reconnectAttempts = 0;
        this.maxReconnectAttempts = 10;
        this.reconnectDelay = 5000;
        this.processedMessages = new Set(); // IDs de mensajes ya procesados
        this.userId = this.generateUserId(); // ID único para este usuario
        
        // Configuración
        this.config = {
            useRealWebSocket: false,
            pollingInterval: 3000, // Consultar cada 3 segundos
            apiBaseUrl: 'https://martin.monteroampsoluciones.com/backend'
        };
        
        this.init();
    }
    
    // Generar un ID único para este usuario/sesión
    generateUserId() {
        let userId = sessionStorage.getItem('alerta_user_id');
        if (!userId) {
            userId = 'user_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9);
            sessionStorage.setItem('alerta_user_id', userId);
        }
        return userId;
    }
    
    init() {
        console.log('🔌 Inicializando Sistema de Notificaciones...');
        console.log('👤 ID Usuario:', this.userId);
        
        // Iniciar polling para recibir alertas de otros usuarios
        this.startPolling();
        
        // Configurar event listeners
        this.setupEventListeners();
    }
    
    /**
     * Configura los event listeners
     */
    setupEventListeners() {
        console.log('🎧 Configurando listeners de notificaciones...');
        
        // Escuchar evento de nueva alerta desde otros módulos
        document.addEventListener('nueva-alerta-local', (event) => {
            console.log('📤 Alerta creada por este usuario:', event.detail);
        });
        
        // Escuchar cuando se cierra notificación de emergencia
        document.addEventListener('click', (event) => {
            if (event.target.closest('.notificacion-emergencia button')) {
                console.log('❌ Notificación de emergencia cerrada');
            }
        });
        
        // También responder a eventos de la app principal
        document.addEventListener('app-ready', () => {
            console.log('✅ App principal lista');
        });
    }
    
    // ================= POLLING PARA RECIBIR ALERTAS DE OTROS =================
    startPolling() {
        console.log('🔄 Iniciando sistema de polling para alertas en tiempo real...');
        
        // Consultar mensajes inmediatamente
        setTimeout(() => this.checkForNewMessages(), 1000);
        
        // Configurar intervalo para polling
        this.pollingInterval = setInterval(() => {
            this.checkForNewMessages();
        }, this.config.pollingInterval);
    }
    
    async checkForNewMessages() {
        try {
            const response = await fetch(`${this.config.apiBaseUrl}/api.php?url=websocket-messages`);
            
            if (!response.ok) {
                throw new Error(`HTTP ${response.status}`);
            }
            
            const result = await response.json();
            
            if (result.success && result.messages && result.messages.length > 0) {
                // Procesar cada mensaje nuevo (sólo los no procesados)
                result.messages.forEach(message => {
                    if (!this.processedMessages.has(message.id)) {
                        this.processedMessages.add(message.id);
                        
                        if (message.type === 'new_alert') {
                            this.handleNewAlert(message.data, message.id);
                        }
                    }
                });
                
                // Limitar tamaño de processedMessages
                if (this.processedMessages.size > 200) {
                    const array = Array.from(this.processedMessages);
                    this.processedMessages = new Set(array.slice(-100));
                }
            }
            
        } catch (error) {
            console.log('⚠️ Error en polling:', error.message);
        }
    }
    
    // ================= MANEJAR NUEVAS ALERTAS DE OTROS USUARIOS =================
    handleNewAlert(alertData, messageId) {
        console.log('🚨 ALERTA RECIBIDA DE OTRO USUARIO:', alertData);
        
        // Verificar si es una alerta reciente (menos de 2 minutos)
        const now = Date.now();
        let alertTime;
        
        if (alertData.timestamp) {
            alertTime = new Date(alertData.timestamp).getTime();
        } else if (alertData.fecha_creacion) {
            alertTime = new Date(alertData.fecha_creacion).getTime();
        } else {
            alertTime = now;
        }
        
        const isRecent = (now - alertTime) < 120000; // 2 minutos
        
        if (!isRecent) {
            console.log('⏳ Alerta muy antigua, ignorando...');
            return;
        }
        
        // IMPORTANTE: Solo mostrar si es una alerta de pánico O si es para todos
        const esPanico = alertData.es_panico || alertData.tipo === 'panico';
        const esParaTodos = alertData.para_todos === true;
        
        if (!esPanico && !esParaTodos) {
            console.log('📨 Alerta normal, solo para el creador');
            return;
        }
        
        // 1. Mostrar notificación emergente si es pánico
        if (esPanico) {
            this.showEmergencyNotification(alertData);
        }
        
        // 2. Mostrar notificación toast para cualquier alerta importante
        if (window.mostrarNotificacion) {
            const tipo = alertData.tipo || 'otro';
            const iconos = {
                'panico': '🚨',
                'robo': '💰',
                'accidente': '🚗',
                'incendio': '🔥',
                'violencia': '⚔️',
                'otro': '🔔'
            };
            
            const icono = iconos[tipo] || '🔔';
            
            // Texto diferente si es de pánico
            let mensaje = `${icono} ${alertData.titulo || 'Nueva alerta'}`;
            if (esPanico) {
                mensaje = `${icono} ¡ALERTA DE PÁNICO! ${alertData.titulo || 'Ayuda necesaria'}`;
            }
            
            // Evitar notificaciones duplicadas para la misma alerta
            const alertKey = `alert_${messageId || alertData.id}`;
            if (!sessionStorage.getItem(alertKey)) {
                const tipoNotif = esPanico ? 'panico' : 'info';
                window.mostrarNotificacion(tipoNotif, mensaje, esPanico ? 10000 : 5000);
                sessionStorage.setItem(alertKey, 'shown');
                
                // Limpiar después de 5 minutos
                setTimeout(() => {
                    sessionStorage.removeItem(alertKey);
                }, 300000);
            }
        }
        
        // 3. Actualizar lista de alertas en tiempo real
        if (typeof window.cargarAlertasRecientes === 'function') {
            // Pequeño delay para asegurar que la BD se actualizó
            setTimeout(() => {
                window.cargarAlertasRecientes();
                console.log('🔄 Lista de alertas actualizada por alerta externa');
            }, 1500);
        }
        
        // 4. Actualizar mapa si hay coordenadas
        if (typeof window.añadirAlertaAlMapa === 'function' && alertData.latitud && alertData.longitud) {
            setTimeout(() => {
                window.añadirAlertaAlMapa(alertData);
                console.log('🗺️ Alerta añadida al mapa');
            }, 2000);
        }
        
        // 5. Disparar evento para que otros módulos reaccionen
        const event = new CustomEvent('nueva-alerta-externa', { 
            detail: {
                ...alertData,
                esExterna: true,
                recibidaEn: new Date().toISOString()
            }
        });
        document.dispatchEvent(event);
        
        // 6. Reproducir sonido de alerta si es pánico
        if (esPanico) {
            this.playAlertSound();
        }
    }
    
    // Mostrar notificación de emergencia en pantalla completa
    showEmergencyNotification(alertData) {
        const emergencia = document.getElementById('notificacion-emergencia');
        if (emergencia) {
            // Actualizar mensaje
            const tituloElem = emergencia.querySelector('.mensaje h3');
            const mensajeElem = emergencia.querySelector('.mensaje p');
            
            if (tituloElem) {
                tituloElem.textContent = '¡EMERGENCIA!';
            }
            
            if (mensajeElem) {
                mensajeElem.textContent = alertData.titulo || 'Alerta de pánico activada por otro usuario';
                
                // Añadir ubicación si existe
                if (alertData.ubicacion) {
                    mensajeElem.innerHTML += `<br><small>📍 ${alertData.ubicacion}</small>`;
                }
                
                // Añadir hora si existe
                if (alertData.fecha_human) {
                    mensajeElem.innerHTML += `<br><small>⏰ ${alertData.fecha_human}</small>`;
                }
            }
            
            // Mostrar con animación
            emergencia.style.display = 'block';
            emergencia.style.animation = 'pulse 1s infinite';
            
            // Auto-ocultar después de 30 segundos (más tiempo para emergencias)
            setTimeout(() => {
                if (emergencia.style.display === 'block') {
                    emergencia.style.display = 'none';
                    emergencia.style.animation = '';
                }
            }, 30000);
        } else {
            // Si no existe el elemento, crearlo dinámicamente
            this.createEmergencyNotification(alertData);
        }
    }
    
    // Crear notificación de emergencia dinámicamente
    createEmergencyNotification(alertData) {
        const emergenciaHTML = `
            <div id="notificacion-emergencia" class="notificacion-emergencia" style="
                position: fixed;
                top: 0;
                left: 0;
                width: 100%;
                background-color: #dc3545;
                color: white;
                padding: 1rem;
                z-index: 2000;
                display: flex;
                justify-content: center;
                align-items: center;
                box-shadow: 0 4px 12px rgba(0, 0, 0, 0.3);
                animation: pulse 1s infinite;
            ">
                <div class="contenido" style="
                    display: flex;
                    align-items: center;
                    max-width: 1200px;
                    width: 100%;
                ">
                    <div class="icono" style="margin-right: 1rem;">
                        <i class="fas fa-exclamation-triangle fa-3x"></i>
                    </div>
                    <div class="mensaje">
                        <h3 style="margin: 0 0 0.5rem 0; font-size: 1.5rem;">¡EMERGENCIA!</h3>
                        <p style="margin: 0;">
                            ${alertData.titulo || 'Alerta de pánico activada'}
                            ${alertData.ubicacion ? `<br><small>📍 ${alertData.ubicacion}</small>` : ''}
                            ${alertData.fecha_human ? `<br><small>⏰ ${alertData.fecha_human}</small>` : ''}
                        </p>
                    </div>
                    <button class="btn btn-light" onclick="this.parentElement.parentElement.style.display='none'" style="
                        background-color: rgba(255, 255, 255, 0.2);
                        border: 1px solid white;
                        color: white;
                        border-radius: 4px;
                        padding: 0.5rem 1rem;
                        margin-left: 1rem;
                        cursor: pointer;
                    ">
                        <i class="fas fa-times"></i>
                    </button>
                </div>
            </div>
        `;
        
        // Añadir al cuerpo
        document.body.insertAdjacentHTML('afterbegin', emergenciaHTML);
        
        // Auto-ocultar después de 30 segundos
        setTimeout(() => {
            const emergencia = document.getElementById('notificacion-emergencia');
            if (emergencia) {
                emergencia.style.display = 'none';
            }
        }, 30000);
    }
    
    // Reproducir sonido de alerta
    playAlertSound() {
        try {
            // Crear audio context para sonido de emergencia
            const audioContext = new (window.AudioContext || window.webkitAudioContext)();
            
            // Crear oscilador para sonido de sirena
            const oscillator = audioContext.createOscillator();
            const gainNode = audioContext.createGain();
            
            oscillator.connect(gainNode);
            gainNode.connect(audioContext.destination);
            
            // Configurar sonido de sirena
            oscillator.frequency.setValueAtTime(800, audioContext.currentTime);
            oscillator.frequency.exponentialRampToValueAtTime(1200, audioContext.currentTime + 0.5);
            oscillator.frequency.exponentialRampToValueAtTime(800, audioContext.currentTime + 1);
            
            oscillator.type = 'sine';
            
            // Configurar volumen (bajo para no molestar demasiado)
            gainNode.gain.setValueAtTime(0.1, audioContext.currentTime);
            gainNode.gain.exponentialRampToValueAtTime(0.01, audioContext.currentTime + 2);
            
            // Reproducir por 2 segundos
            oscillator.start(audioContext.currentTime);
            oscillator.stop(audioContext.currentTime + 2);
            
        } catch (e) {
            console.log('🔇 Audio no disponible o bloqueado por el navegador');
        }
    }
    
    // ================= ENVIAR MENSAJE (para futuras expansiones) =================
    send(data) {
        console.log('📤 Enviando mensaje:', data);
        // Por ahora solo recibimos, no enviamos
        return false;
    }
    
    // ================= DESTRUIR =================
    destroy() {
        if (this.pollingInterval) {
            clearInterval(this.pollingInterval);
        }
        
        console.log(' Sistema de notificaciones detenido');
    }
}

// ================= INICIALIZACIÓN GLOBAL =================
let alertWebSocket = null;

document.addEventListener('DOMContentLoaded', function() {
    // Iniciar después de que cargue la app principal
    setTimeout(() => {
        try {
            alertWebSocket = new AlertWebSocketClient();
            window.alertWebSocket = alertWebSocket;
            
            console.log(' Sistema de notificaciones en tiempo real iniciado');
            console.log('📡 Escuchando alertas de otros usuarios...');
            
            // Disparar evento para indicar que WebSocket está listo
            document.dispatchEvent(new CustomEvent('websocket-ready', {
                detail: { tipo: 'polling', intervalo: 3000 }
            }));
            
        } catch (error) {
            console.error(' Error inicializando notificaciones:', error);
        }
    }, 2000);
});

// Exportar para uso global
window.AlertWebSocketClient = AlertWebSocketClient;

// Funciones de utilidad
window.iniciarNotificaciones = function() {
    if (!alertWebSocket) {
        alertWebSocket = new AlertWebSocketClient();
    }
    return alertWebSocket;
};

window.detenerNotificaciones = function() {
    if (alertWebSocket) {
        alertWebSocket.destroy();
        alertWebSocket = null;
    }
};

// Función para probar el sistema
window.probarNotificacion = function(tipo = 'panico') {
    const alertaPrueba = {
        id: 'test_' + Date.now(),
        tipo: tipo,
        titulo: tipo === 'panico' ? ' PRUEBA: Alerta de Pánico' : 'PRUEBA: Alerta Normal',
        descripcion: 'Esta es una alerta de prueba del sistema',
        ubicacion: 'Ubicación de prueba',
        prioridad: 'alta',
        timestamp: new Date().toISOString(),
        es_panico: tipo === 'panico',
        para_todos: true
    };
    
    if (alertWebSocket) {
        alertWebSocket.handleNewAlert(alertaPrueba, 'test_id');
    } else {
        console.log(' Sistema de notificaciones no iniciado');
    }
};