<?php
class AlertasModel {
    private $conn;
    private $table = 'alertas';
    
    public function __construct($db) {
        $this->conn = $db;
    }
    
    // Crear tabla si no existe
    public function createTableIfNotExists() {
        $sql = "CREATE TABLE IF NOT EXISTS `alertas` (
            `id` INT AUTO_INCREMENT PRIMARY KEY,
            `tipo` VARCHAR(50) NOT NULL,
            `titulo` VARCHAR(255) NOT NULL,
            `descripcion` TEXT NOT NULL,
            `ubicacion` VARCHAR(500) NOT NULL,
            `latitud` DECIMAL(10, 8),
            `longitud` DECIMAL(11, 8),
            `prioridad` ENUM('baja', 'media', 'alta') DEFAULT 'media',
            `foto_url` VARCHAR(500),
            `estado` ENUM('activa', 'atendida', 'cancelada') DEFAULT 'activa',
            `fecha_creacion` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            INDEX idx_fecha (fecha_creacion),
            INDEX idx_prioridad (prioridad),
            INDEX idx_tipo (tipo)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        try {
            $stmt = $this->conn->prepare($sql);
            $stmt->execute();
            return true;
        } catch(PDOException $e) {
            return false;
        }
    }
    
    // Obtener alertas recientes
    public function getAlertasRecientes($limit = 10) {
        $sql = "SELECT * FROM `alertas` 
                ORDER BY `fecha_creacion` DESC 
                LIMIT :limit";
        
        $stmt = $this->conn->prepare($sql);
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    // Crear nueva alerta
    public function createAlerta($data) {
        $sql = "INSERT INTO `alertas` 
                (tipo, titulo, descripcion, ubicacion, latitud, longitud, prioridad, foto_url) 
                VALUES 
                (:tipo, :titulo, :descripcion, :ubicacion, :latitud, :longitud, :prioridad, :foto_url)";
        
        $stmt = $this->conn->prepare($sql);
        
        // Manejar foto (en una implementación real, subirías el archivo)
        $foto_url = null;
        if(isset($data['foto']) && $data['foto'] && $data['foto']['error'] === 0) {
            $foto_url = '/uploads/' . basename($data['foto']['name']);
        }
        
        $stmt->bindValue(':tipo', $data['tipo'] ?? 'otro');
        $stmt->bindValue(':titulo', $data['titulo'] ?? 'Sin título');
        $stmt->bindValue(':descripcion', $data['descripcion'] ?? '');
        $stmt->bindValue(':ubicacion', $data['ubicacion'] ?? 'Ubicación no especificada');
        $stmt->bindValue(':latitud', $data['latitud'] ?? null);
        $stmt->bindValue(':longitud', $data['longitud'] ?? null);
        $stmt->bindValue(':prioridad', $data['prioridad'] ?? 'media');
        $stmt->bindValue(':foto_url', $foto_url);
        
        try {
            $stmt->execute();
            return [
                'success' => true,
                'id' => $this->conn->lastInsertId(),
                'message' => 'Alerta creada exitosamente'
            ];
        } catch(PDOException $e) {
            return [
                'success' => false,
                'message' => 'Error al crear alerta: ' . $e->getMessage()
            ];
        }
    }
    
    // Crear alerta de pánico
    public function createAlertaPanico($latitud, $longitud) {
        $sql = "INSERT INTO `alertas` 
                (tipo, titulo, descripcion, ubicacion, latitud, longitud, prioridad) 
                VALUES 
                ('panico', 'BOTÓN DE PÁNICO ACTIVADO', 
                 'Usuario ha activado el botón de pánico. Se requiere asistencia inmediata.',
                 :ubicacion, :latitud, :longitud, 'alta')";
        
        $stmt = $this->conn->prepare($sql);
        
        $ubicacion = "Lat: $latitud, Lng: $longitud";
        
        $stmt->bindValue(':ubicacion', $ubicacion);
        $stmt->bindValue(':latitud', $latitud);
        $stmt->bindValue(':longitud', $longitud);
        
        try {
            $stmt->execute();
            return [
                'success' => true,
                'id' => $this->conn->lastInsertId(),
                'message' => 'Alerta de pánico creada'
            ];
        } catch(PDOException $e) {
            return [
                'success' => false,
                'message' => 'Error: ' . $e->getMessage()
            ];
        }
    }
}
?>