// ============================================
// MANEJO DE ALERTAS RECIENTES - SOLO BD
// ============================================

let intervaloAutoRefresh = null;

/**
 * Obtiene la URL base de la API
 */
function getApiBaseUrl() {
    return window.API_BASE_URL || 'https://martin.monteroampsoluciones.com/backend';
}

/**
 * Carga y muestra las alertas recientes DESDE LA BD
 */
async function cargarAlertasRecientes(mostrarNotificacion = false) {
    const container = document.getElementById('alerts-container');
    
    if (!container) {
        console.error(' Contenedor de alertas no encontrado');
        return;
    }
    
    // Mostrar estado de carga
    container.innerHTML = `
        <div style="grid-column: 1 / -1; padding: 50px; text-align: center;">
            <i class="fas fa-spinner fa-spin fa-2x" style="color: #3498db;"></i>
            <p style="margin-top: 15px; color: #7f8c8d;">Cargando alertas desde la base de datos...</p>
        </div>
    `;
    
    try {
        console.log(' Solicitando alertas');
        
       // En la función cargarAlertasRecientes() - línea ~36
const respuesta = await fetch(`${getApiBaseUrl()}/api.php?url=alertas`, {
    method: 'GET',
    headers: {
        'Accept': 'application/json',
        'Cache-Control': 'no-cache'
    },
    credentials: 'omit' 
});
        
        console.log(' Estado respuesta:', respuesta.status);
        
        if (!respuesta.ok) {
            throw new Error(`Error HTTP ${respuesta.status}: No se pudo conectar al servidor`);
        }
        
        const resultado = await respuesta.json();
        
        if (resultado.success && resultado.data) {
            mostrarAlertasDesdeBD(resultado.data, container);
            
            if (mostrarNotificacion && window.mostrarNotificacion) {
                window.mostrarNotificacion('success', `Se cargaron ${resultado.data.length} alertas`, 3000);
            }
            
            // Actualizar mapa si existe
            if (typeof cargarAlertasEnMapa === 'function') {
                setTimeout(() => cargarAlertasEnMapa(), 500);
            }
            
        } else {
            throw new Error(resultado.message || 'Error en los datos recibidos');
        }
        
    } catch (error) {
        console.error(' Error cargando alertas:', error);
        
        // Mostrar mensaje de error, NO datos de ejemplo
        container.innerHTML = `
            <div style="grid-column: 1 / -1; padding: 50px; text-align: center; background: #fff5f5; border-radius: 10px;">
                <i class="fas fa-exclamation-triangle fa-2x" style="color: #e74c3c; margin-bottom: 15px;"></i>
                <h3 style="color: #c0392b;">Error al cargar alertas</h3>
                <p style="color: #7f8c8d;">${error.message}</p>
                <p style="color: #95a5a6; margin-top: 10px;">Verifica la conexión con el servidor</p>
                <button onclick="cargarAlertasRecientes(true)" style="
                    margin-top: 20px;
                    padding: 10px 20px;
                    background: #3498db;
                    color: white;
                    border: none;
                    border-radius: 5px;
                    cursor: pointer;
                ">
                    <i class="fas fa-redo"></i> Reintentar
                </button>
            </div>
        `;
        
        if (window.mostrarNotificacion) {
            window.mostrarNotificacion('error', 'No se pudieron cargar las alertas: ' + error.message, 5000);
        }
    }
}

/**
 * Muestra alertas desde la base de datos
 */
function mostrarAlertasDesdeBD(alertas, container) {
    if (!alertas || alertas.length === 0) {
        container.innerHTML = `
            <div style="grid-column: 1 / -1; padding: 50px; text-align: center; background: #f8f9fa; border-radius: 10px;">
                <i class="fas fa-inbox fa-2x" style="color: #bdc3c7; margin-bottom: 15px;"></i>
                <h3 style="color: #7f8c8d;">No hay alertas registradas</h3>
                <p style="color: #95a5a6;">Sé el primero en reportar un incidente</p>
            </div>
        `;
        return;
    }
    
    let html = '';
    
    alertas.forEach((alerta) => {
    // Formatear fecha - Ajustar a hora de Perú
const fecha = new Date(alerta.fecha_creacion);
// Añadir el offset de Perú (UTC-5)
const offsetPeru = -5 * 60; // -5 horas en minutos
const fechaPeru = new Date(fecha.getTime() + (offsetPeru * 60 * 1000));
        
const fechaFormateada = fechaPeru.toLocaleDateString('es-PE', {
    day: '2-digit',
    month: '2-digit',
    year: 'numeric',
    hour: '2-digit',
    minute: '2-digit',
    timeZone: 'America/Lima'
});
        
        // Icono según tipo
        const iconos = {
            'panico': 'fa-exclamation-triangle',
            'robo': 'fa-gem',
            'accidente': 'fa-car-crash',
            'incendio': 'fa-fire',
            'violencia': 'fa-user-injured',
            'otro': 'fa-exclamation-circle'
        };
        
        const icono = iconos[alerta.tipo] || 'fa-bell';
        
        // Color según tipo
        const colores = {
            'panico': '#c0392b',
            'robo': '#e74c3c',
            'accidente': '#f39c12',
            'incendio': '#e67e22',
            'violencia': '#8e44ad',
            'otro': '#3498db'
        };
        
        const color = colores[alerta.tipo] || '#3498db';
        
        html += `
            <div class="alerta-item" style="
                background: white;
                border-radius: 10px;
                padding: 20px;
                margin-bottom: 15px;
                box-shadow: 0 2px 10px rgba(0,0,0,0.1);
                border-left: 4px solid ${color};
            ">
                <div style="display: flex; justify-content: space-between; align-items: flex-start; margin-bottom: 15px;">
                    <div style="display: flex; align-items: center; gap: 10px;">
                        <div style="
                            width: 40px;
                            height: 40px;
                            border-radius: 50%;
                            background: ${color};
                            color: white;
                            display: flex;
                            align-items: center;
                            justify-content: center;
                            font-size: 1.2rem;
                        ">
                            <i class="fas ${icono}"></i>
                        </div>
                        <div>
                            <div style="font-weight: bold; color: #2c3e50; text-transform: uppercase; font-size: 0.9rem;">
                                ${alerta.tipo}
                            </div>
                            <div style="font-size: 0.8rem; color: #7f8c8d;">
                                ID: ${alerta.id}
                            </div>
                        </div>
                    </div>
                    
                    <div style="
                        padding: 5px 10px;
                        border-radius: 20px;
                        font-size: 0.8rem;
                        font-weight: bold;
                        background: ${alerta.prioridad === 'alta' ? '#ffebee' : 
                                    alerta.prioridad === 'baja' ? '#e8f5e9' : '#fff3e0'};
                        color: ${alerta.prioridad === 'alta' ? '#c62828' : 
                                alerta.prioridad === 'baja' ? '#2e7d32' : '#ef6c00'};
                    ">
                        ${alerta.prioridad ? alerta.prioridad.toUpperCase() : 'MEDIA'}
                    </div>
                </div>
                
                <h3 style="margin: 0 0 10px 0; color: #2c3e50;">
                    ${alerta.titulo || 'Sin título'}
                </h3>
                
                <p style="margin: 0 0 15px 0; color: #546e7a; line-height: 1.5;">
                    ${alerta.descripcion || 'Sin descripción'}
                </p>
                
                <div style="display: flex; justify-content: space-between; align-items: center; margin-top: 15px; padding-top: 15px; border-top: 1px solid #eee;">
                    <div style="display: flex; align-items: center; gap: 5px; color: #7f8c8d; font-size: 0.9rem;">
                        <i class="fas fa-map-marker-alt"></i>
                        <span>${alerta.ubicacion || 'Ubicación no especificada'}</span>
                    </div>
                    
                    <div style="display: flex; align-items: center; gap: 5px; color: #95a5a6; font-size: 0.85rem;">
                        <i class="far fa-clock"></i>
                        <span>${fechaFormateada}</span>
                        ${alerta.tiempo_relativo ? `<span style="color: #bdc3c7;">(${alerta.tiempo_relativo})</span>` : ''}
                    </div>
                </div>
                
                ${alerta.latitud && alerta.longitud ? `
                <div style="margin-top: 15px;">
                    <button onclick="verEnMapa(${alerta.latitud}, ${alerta.longitud}, '${alerta.titulo ? alerta.titulo.replace(/'/g, "\\'") : ''}')" 
                            style="
                        padding: 8px 15px;
                        background: #3498db;
                        color: white;
                        border: none;
                        border-radius: 5px;
                        cursor: pointer;
                        font-size: 0.9rem;
                    ">
                        <i class="fas fa-map"></i> Ver en mapa
                    </button>
                </div>
                ` : ''}
            </div>
        `;
    });
    
    // Botón para recargar
    html += `
        <div style="grid-column: 1 / -1; text-align: center; margin-top: 20px;">
            <button onclick="cargarAlertasRecientes(true)" style="
                padding: 10px 20px;
                background: #6c757d;
                color: white;
                border: none;
                border-radius: 5px;
                cursor: pointer;
            ">
                <i class="fas fa-sync-alt"></i> Actualizar alertas
            </button>
            <div style="margin-top: 8px; color: #95a5a6; font-size: 0.9rem;">
                Última actualización: ${new Date().toLocaleTimeString('es-PE')}
            </div>
        </div>
    `;
    
    container.innerHTML = html;
    console.log(` ${alertas.length} alertas cargadas`);
}

/**
 * Ver alerta en el mapa
 */
function verEnMapa(latitud, longitud, titulo) {
    if (typeof window.centrarMapaEn === 'function') {
        window.centrarMapaEn(latitud, longitud, titulo);
        
        if (typeof window.mostrarNotificacion === 'function') {
            window.mostrarNotificacion('info', `Mapa centrado en: ${titulo}`, 3000);
        }
    }
}

/**
 * Inicia la actualización automática
 */
function iniciarAutoRefresh(intervaloMinutos = 2) {
    // Detener intervalo existente
    if (intervaloAutoRefresh) {
        clearInterval(intervaloAutoRefresh);
    }
    
    // Convertir minutos a milisegundos
    const intervaloMs = intervaloMinutos * 60 * 1000;
    
    // Configurar intervalo
    intervaloAutoRefresh = setInterval(() => {
        console.log(' Actualización automática de alertas');
        cargarAlertasRecientes(false);
    }, intervaloMs);
    
    console.log(` Auto-refresh configurado cada ${intervaloMinutos} minuto(s)`);
}

/**
 * Detiene la actualización automática
 */
function detenerAutoRefresh() {
    if (intervaloAutoRefresh) {
        clearInterval(intervaloAutoRefresh);
        intervaloAutoRefresh = null;
        console.log(' Auto-refresh detenido');
    }
}

// ============================================
// INICIALIZACIÓN
// ============================================

// Iniciar auto-refresh cuando la página cargue
document.addEventListener('DOMContentLoaded', function() {
    // Esperar un poco después de la carga inicial
    setTimeout(() => iniciarAutoRefresh(2), 10000);
});

// ============================================
// EXPORTACIÓN PARA USO GLOBAL
// ============================================

window.cargarAlertasRecientes = cargarAlertasRecientes;
window.verEnMapa = verEnMapa;
window.iniciarAutoRefresh = iniciarAutoRefresh;
window.detenerAutoRefresh = detenerAutoRefresh;
window.getApiBaseUrl = getApiBaseUrl;

console.log(' alerts.js cargado');