<?php
// ============================================
// PRUEBAS COMPLETAS DEL SISTEMA DE ALERTAS
// ============================================

echo "<!DOCTYPE html>
<html lang='es'>
<head>
    <meta charset='UTF-8'>
    <title>🔍 Pruebas del Sistema de Alertas VES</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; }
        .test { margin: 15px 0; padding: 10px; border-left: 4px solid #ccc; }
        .success { border-color: #2ecc71; background: #d5f4e6; }
        .warning { border-color: #f39c12; background: #fef5e7; }
        .error { border-color: #e74c3c; background: #fadbd8; }
        .info { border-color: #3498db; background: #ebf5fb; }
        h1, h2, h3 { color: #2c3e50; }
        pre { background: #f8f9fa; padding: 10px; border-radius: 5px; }
        .btn { display: inline-block; padding: 8px 15px; margin: 5px; background: #3498db; color: white; text-decoration: none; border-radius: 4px; }
        .btn:hover { opacity: 0.9; }
    </style>
</head>
<body>
    <h1>🔍 Sistema de Pruebas - Alerta VES</h1>
    <p><strong>Fecha:</strong> " . date('d/m/Y H:i:s') . " | <strong>PHP:</strong> " . phpversion() . "</p>
";

// ============================================
// CONFIGURACIÓN
// ============================================
$base_url = 'https://martin.monteroampsoluciones.com';
$api_url = $base_url . '/backend/microservices/alertas/';
$api_recientes = $api_url . 'recientes';

// ============================================
// 1. PRUEBA: CONEXIÓN AL SERVIDOR
// ============================================
echo "<div class='test info'><h3>1. 🔌 Conexión al servidor</h3>";

$server_online = @fsockopen('martin.monteroampsoluciones.com', 443, $errno, $errstr, 10);
if ($server_online) {
    echo "<p class='success'>✅ Servidor accesible (martin.monteroampsoluciones.com:443)</p>";
    fclose($server_online);
} else {
    echo "<p class='error'>❌ Servidor no accesible: $errstr ($errno)</p>";
}
echo "</div>";

// ============================================
// 2. PRUEBA: PHP FUNCIONANDO
// ============================================
echo "<div class='test info'><h3>2. 🐘 PHP Funcionando</h3>";
echo "<p><strong>Versión PHP:</strong> " . phpversion() . "</p>";
echo "<p><strong>Memoria:</strong> " . ini_get('memory_limit') . "</p>";
echo "<p><strong>Zona horaria:</strong> " . date_default_timezone_get() . "</p>";

// Verificar funciones requeridas
$required_functions = ['json_encode', 'json_decode', 'date_default_timezone_set', 'header'];
$missing_functions = [];
foreach ($required_functions as $func) {
    if (!function_exists($func)) {
        $missing_functions[] = $func;
    }
}

if (empty($missing_functions)) {
    echo "<p class='success'>✅ Todas las funciones PHP requeridas están disponibles</p>";
} else {
    echo "<p class='error'>❌ Funciones faltantes: " . implode(', ', $missing_functions) . "</p>";
}
echo "</div>";

// ============================================
// 3. PRUEBA: CONEXIÓN A BASE DE DATOS
// ============================================
echo "<div class='test info'><h3>3. 💾 Conexión a Base de Datos</h3>";

try {
    $host = 'localhost';
    $dbname = 'monteroa_alerta_ves';
    $username = 'monteroa_alerta_ves';
    $password = 'AlertaVES2025!';
    
    $conn = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8", $username, $password);
    $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    echo "<p class='success'>✅ Conexión a MySQL exitosa</p>";
    
    // Verificar tabla alertas
    $stmt = $conn->query("SHOW TABLES LIKE 'alertas'");
    if ($stmt->rowCount() > 0) {
        echo "<p class='success'>✅ Tabla 'alertas' existe</p>";
        
        // Contar registros
        $count = $conn->query("SELECT COUNT(*) as total FROM alertas")->fetch()['total'];
        echo "<p><strong>Total alertas en BD:</strong> $count</p>";
        
        // Verificar estructura
        $columns = $conn->query("DESCRIBE alertas")->fetchAll(PDO::FETCH_COLUMN);
        $required_columns = ['id', 'tipo', 'titulo', 'descripcion', 'ubicacion', 'prioridad', 'fecha_creacion'];
        $missing_columns = array_diff($required_columns, $columns);
        
        if (empty($missing_columns)) {
            echo "<p class='success'>✅ Estructura de tabla correcta</p>";
        } else {
            echo "<p class='warning'>⚠️ Columnas faltantes: " . implode(', ', $missing_columns) . "</p>";
        }
        
    } else {
        echo "<p class='error'>❌ Tabla 'alertas' NO existe</p>";
    }
    
} catch (PDOException $e) {
    echo "<p class='error'>❌ Error de conexión a BD: " . $e->getMessage() . "</p>";
}
echo "</div>";

// ============================================
// 4. PRUEBA: API REST - ENDPOINT GET
// ============================================
echo "<div class='test info'><h3>4. 📡 API REST - GET Alertas</h3>";
echo "<p><strong>URL probada:</strong> <a href='$api_recientes' target='_blank'>$api_recientes</a></p>";

$context = stream_context_create([
    'ssl' => [
        'verify_peer' => false,
        'verify_peer_name' => false,
    ],
    'http' => [
        'method' => 'GET',
        'header' => "Accept: application/json\r\n"
    ]
]);

$response = @file_get_contents($api_recientes, false, $context);
$http_response_header_info = $http_response_header ?? [];

if ($response !== false) {
    echo "<p class='success'>✅ API respondió correctamente</p>";
    
    // Mostrar headers de respuesta
    if (!empty($http_response_header_info)) {
        echo "<p><strong>Headers:</strong></p><pre>";
        foreach ($http_response_header_info as $header) {
            echo htmlspecialchars($header) . "\n";
        }
        echo "</pre>";
    }
    
    // Decodificar JSON
    $data = json_decode($response, true);
    
    if (json_last_error() === JSON_ERROR_NONE) {
        echo "<p class='success'>✅ Respuesta JSON válida</p>";
        
        echo "<p><strong>Estructura de respuesta:</strong></p><pre>";
        echo json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        echo "</pre>";
        
        // Validar estructura esperada
        if (isset($data['success']) && $data['success'] === true) {
            echo "<p class='success'>✅ Estructura de respuesta correcta (success: true)</p>";
            
            if (isset($data['data']) || isset($data['alertas'])) {
                $alertas = $data['data'] ?? $data['alertas'] ?? [];
                $count = is_array($alertas) ? count($alertas) : 0;
                echo "<p><strong>Alertas recibidas:</strong> $count</p>";
                
                // Mostrar primera alerta como ejemplo
                if ($count > 0 && is_array($alertas)) {
                    $first = reset($alertas);
                    echo "<p><strong>Ejemplo de alerta:</strong></p><pre>";
                    print_r($first);
                    echo "</pre>";
                    
                    // Verificar fecha formateada
                    if (isset($first['fecha_formateada'])) {
                        echo "<p class='success'>✅ Fecha formateada presente: " . htmlspecialchars($first['fecha_formateada']) . "</p>";
                    }
                }
            }
        } else {
            echo "<p class='warning'>⚠️ Respuesta indica error: " . ($data['message'] ?? 'Desconocido') . "</p>";
        }
        
    } else {
        echo "<p class='error'>❌ JSON inválido: " . json_last_error_msg() . "</p>";
        echo "<p><strong>Respuesta cruda:</strong></p><pre>" . htmlspecialchars($response) . "</pre>";
    }
    
} else {
    echo "<p class='error'>❌ No se pudo conectar a la API</p>";
    
    // Intentar con curl como alternativa
    if (function_exists('curl_init')) {
        echo "<p>Intentando con cURL...</p>";
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $api_recientes);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Accept: application/json']);
        
        $curl_response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        
        if ($curl_response !== false) {
            echo "<p><strong>cURL HTTP Code:</strong> $http_code</p>";
            echo "<p><strong>Respuesta:</strong></p><pre>" . htmlspecialchars($curl_response) . "</pre>";
        } else {
            echo "<p class='error'>❌ cURL también falló: " . curl_error($ch) . "</p>";
        }
        
        curl_close($ch);
    }
}
echo "</div>";

// ============================================
// 5. PRUEBA: API REST - ENDPOINT POST
// ============================================
echo "<div class='test info'><h3>5. 📨 API REST - POST Crear Alerta</h3>";

if (function_exists('curl_init')) {
    $test_data = [
        'tipo' => 'test',
        'titulo' => 'Prueba automática ' . date('H:i:s'),
        'descripcion' => 'Esta es una alerta de prueba generada automáticamente',
        'ubicacion' => 'Ubicación de prueba',
        'prioridad' => 'media',
        'latitud' => -12.0464,
        'longitud' => -77.0428
    ];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $api_url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($test_data));
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'Accept: application/json'
    ]);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    
    echo "<p><strong>Datos enviados:</strong></p><pre>";
    echo json_encode($test_data, JSON_PRETTY_PRINT);
    echo "</pre>";
    
    echo "<p><strong>Código HTTP:</strong> $http_code</p>";
    
    if ($response !== false) {
        $data = json_decode($response, true);
        
        if (json_last_error() === JSON_ERROR_NONE) {
            echo "<p><strong>Respuesta:</strong></p><pre>";
            echo json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
            echo "</pre>";
            
            if (isset($data['success']) && $data['success'] === true) {
                echo "<p class='success'>✅ POST funcionando correctamente</p>";
                echo "<p><strong>ID creado:</strong> " . ($data['id'] ?? 'N/A') . "</p>";
            } else {
                echo "<p class='warning'>⚠️ POST respondió con error: " . ($data['message'] ?? 'Desconocido') . "</p>";
            }
        } else {
            echo "<p class='error'>❌ Respuesta JSON inválida</p>";
            echo "<pre>" . htmlspecialchars($response) . "</pre>";
        }
    } else {
        echo "<p class='error'>❌ Error en la solicitud POST: " . curl_error($ch) . "</p>";
    }
    
    curl_close($ch);
} else {
    echo "<p class='warning'>⚠️ cURL no disponible, no se puede probar POST</p>";
}
echo "</div>";

// ============================================
// 6. PRUEBA: CORS HEADERS
// ============================================
echo "<div class='test info'><h3>6. 🔗 Headers CORS</h3>";

$headers = @get_headers($api_recientes, 1);
if ($headers !== false) {
    $cors_headers = [
        'Access-Control-Allow-Origin' => false,
        'Access-Control-Allow-Methods' => false,
        'Access-Control-Allow-Headers' => false
    ];
    
    foreach ($headers as $key => $value) {
        $key_lower = strtolower($key);
        if (strpos($key_lower, 'access-control') !== false) {
            echo "<p><strong>$key:</strong> $value</p>";
            
            if (strpos($key_lower, 'origin') !== false) $cors_headers['Access-Control-Allow-Origin'] = true;
            if (strpos($key_lower, 'methods') !== false) $cors_headers['Access-Control-Allow-Methods'] = true;
            if (strpos($key_lower, 'headers') !== false) $cors_headers['Access-Control-Allow-Headers'] = true;
        }
    }
    
    $all_cors_present = !in_array(false, $cors_headers, true);
    if ($all_cors_present) {
        echo "<p class='success'>✅ Headers CORS configurados correctamente</p>";
    } else {
        echo "<p class='warning'>⚠️ Faltan algunos headers CORS</p>";
    }
} else {
    echo "<p class='error'>❌ No se pudieron obtener headers</p>";
}
echo "</div>";

// ============================================
// 7. PRUEBA: ZONA HORARIA
// ============================================
echo "<div class='test info'><h3>7. 🕐 Zona Horaria</h3>";

date_default_timezone_set('America/Lima');
$hora_peru = date('H:i:s');
$hora_utc = gmdate('H:i:s');

echo "<p><strong>Hora Perú (America/Lima):</strong> $hora_peru</p>";
echo "<p><strong>Hora UTC:</strong> $hora_utc</p>";

$diferencia = (strtotime($hora_utc) - strtotime($hora_peru)) / 3600;
echo "<p><strong>Diferencia:</strong> $diferencia horas</p>";

if (abs($diferencia - 5) < 0.5) {
    echo "<p class='success'>✅ Zona horaria configurada correctamente (GMT-5)</p>";
} else {
    echo "<p class='error'>❌ Zona horaria incorrecta, debería ser GMT-5 (5 horas de diferencia)</p>";
}
echo "</div>";

// ============================================
// 8. PRUEBA: FRONTEND ACCESIBLE
// ============================================
echo "<div class='test info'><h3>8. 🌐 Frontend Accesible</h3>";

$frontend_url = $base_url . '/';
$frontend_response = @file_get_contents($frontend_url, false, stream_context_create([
    'ssl' => ['verify_peer' => false]
]));

if ($frontend_response !== false) {
    echo "<p class='success'>✅ Frontend accesible en <a href='$frontend_url' target='_blank'>$frontend_url</a></p>";
    
    // Verificar elementos clave en el HTML
    if (strpos($frontend_response, 'Alerta VES') !== false) {
        echo "<p class='success'>✅ Título 'Alerta VES' encontrado</p>";
    }
    
    if (strpos($frontend_response, 'alertaForm') !== false) {
        echo "<p class='success'>✅ Formulario de alertas encontrado</p>";
    }
    
    if (strpos($frontend_response, 'alerts-container') !== false) {
        echo "<p class='success'>✅ Contenedor de alertas encontrado</p>";
    }
    
    if (strpos($frontend_response, 'btnPanico') !== false) {
        echo "<p class='success'>✅ Botón de pánico encontrado</p>";
    }
    
} else {
    echo "<p class='error'>❌ Frontend no accesible</p>";
}
echo "</div>";

// ============================================
// RESUMEN FINAL
// ============================================
echo "<div class='test' style='background:#f8f9fa;padding:20px;margin-top:30px;border:2px solid #3498db;'>
    <h2>📊 RESUMEN DE PRUEBAS</h2>
    <p><strong>URLs importantes:</strong></p>
    <ul>
        <li><a href='$base_url/' target='_blank'>Frontend Principal</a></li>
        <li><a href='$api_url' target='_blank'>API Raíz</a></li>
        <li><a href='$api_recientes' target='_blank'>API Alertas Recientes</a></li>
    </ul>
    <p><strong>Acciones recomendadas:</strong></p>
    <ol>
        <li>Verificar que todas las pruebas muestren ✅ (éxito)</li>
        <li>Si hay ❌ (errores), corregirlos en orden</li>
        <li>Probar el sistema manualmente después de las correcciones</li>
        <li>Ejecutar estas pruebas después de cada cambio importante</li>
    </ol>
    
    <div style='margin-top:20px;'>
        <a href='test-database.php' class='btn'>🧪 Más Pruebas de BD</a>
        <a href='test-security.php' class='btn' style='background:#e74c3c;'>🛡️ Pruebas de Seguridad</a>
        <a href='test-performance.php' class='btn' style='background:#f39c12;'>⚡ Pruebas de Rendimiento</a>
        <button onclick='location.reload()' class='btn' style='background:#2ecc71;'>🔄 Ejecutar Pruebas Nuevamente</button>
    </div>
</div>";

echo "</body></html>";
?>