// ============================================
// SISTEMA DE MAPAS - RUTAS CORREGIDAS
// ============================================

let mapa = null;
let marcadores = [];
let capaAlertas = null;

/**
 * Inicializa el mapa Leaflet
 */
function inicializarMapa() {
    console.log(' Inicializando mapa...');
    
    const contenedorMapa = document.getElementById('map');
    if(!contenedorMapa) {
        console.warn('Contenedor #map no encontrado');
        return;
    }
    
    // Coordenadas por defecto (Lima, Perú)
    const latitudDefault = -12.0464;
    const longitudDefault = -77.0428;
    const zoomDefault = 12;
    
    try {
        // Crear mapa
        mapa = L.map('map').setView([latitudDefault, longitudDefault], zoomDefault);
        
        // Añadir capa de tiles (OpenStreetMap)
        L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
            attribution: '© OpenStreetMap contributors',
            maxZoom: 19
        }).addTo(mapa);
        
        // Capa para agrupar alertas
        capaAlertas = L.layerGroup().addTo(mapa);
        
        // Intentar centrar en ubicación del usuario
        centrarEnUbicacionUsuario();
        
        
        // Cargar alertas en el mapa
        setTimeout(cargarAlertasEnMapa, 1000);
        
    } catch(error) {
        console.error(' Error inicializando mapa:', error);
        mostrarMapaEjemplo(contenedorMapa);
    }
}

/**
 * Centra el mapa en la ubicación del usuario
 */
function centrarEnUbicacionUsuario() {
    if(!navigator.geolocation) {
        console.warn('Geolocalización no soportada');
        return;
    }
    
    navigator.geolocation.getCurrentPosition(
        (posicion) => {
            const lat = posicion.coords.latitude;
            const lng = posicion.coords.longitude;
            
            mapa.setView([lat, lng], 14);
            
            // Añadir marcador de ubicación actual
            L.marker([lat, lng], {
                icon: L.divIcon({
                    className: 'marcador-usuario',
                    html: '<i class="fas fa-user-circle fa-2x" style="color:#3498db;"></i>',
                    iconSize: [30, 30],
                    iconAnchor: [15, 30]
                })
            })
            .addTo(mapa)
            .bindPopup('<b>Tu ubicación actual</b>')
            .openPopup();
            
            
        },
        (error) => {
            console.warn('No se pudo obtener ubicación:', error.message);
        },
        { enableHighAccuracy: true, timeout: 10000, maximumAge: 0 }
    );
}

/**
 * Centra el mapa en coordenadas específicas
 */
function centrarMapaEn(lat, lng, titulo = 'Ubicación') {
    if(!mapa) {
        console.warn('Mapa no inicializado');
        return;
    }
    
    if(!lat || !lng || lat === 0 || lng === 0) {
        console.warn('Coordenadas inválidas:', lat, lng);
        return;
    }
    
    mapa.setView([lat, lng], 16);
    
    // Añadir marcador temporal
    const marcador = L.marker([lat, lng]).addTo(mapa);
    marcador.bindPopup(`<b>${titulo}</b><br>Lat: ${lat.toFixed(6)}, Lng: ${lng.toFixed(6)}`).openPopup();
    
    // Eliminar después de 10 segundos
    setTimeout(() => {
        if(mapa && marcador) {
            mapa.removeLayer(marcador);
        }
    }, 10000);
    
    console.log(` Mapa centrado en: ${titulo} (${lat}, ${lng})`);
}

/**
 * Carga y muestra las alertas en el mapa
 */
async function cargarAlertasEnMapa() {
    if(!mapa || !capaAlertas) {
        console.warn('Mapa no listo para cargar alertas');
        return;
    }
    
    try {
        // Limpiar marcadores anteriores
        capaAlertas.clearLayers();
        marcadores = [];
        
        // RUTA CORREGIDA: Usar API Gateway
        const respuesta = await fetch('https://martin.monteroampsoluciones.com/backend/api.php?url=alertas');
        
        if(!respuesta.ok) {
            throw new Error(`API: ${respuesta.status}`);
        }
        
        const resultado = await respuesta.json();
        
        if(!resultado.success || !resultado.data) {
            // Si no hay datos, usar ejemplo
            cargarAlertasEjemploEnMapa();
            return;
        }
        
        const alertas = resultado.data;
        console.log(` Mostrando ${alertas.length} alertas en el mapa`);
        
        // Colores por tipo de alerta
        const colores = {
            'robo': 'red',
            'accidente': 'orange',
            'incendio': 'darkred',
            'violencia': 'purple',
            'panico': 'black',
            'otro': 'blue'
        };
        
        // Íconos personalizados
        const iconos = {
            'robo': 'fa-gem',
            'accidente': 'fa-car-crash',
            'incendio': 'fa-fire',
            'violencia': 'fa-user-injured',
            'panico': 'fa-exclamation-triangle',
            'otro': 'fa-exclamation-circle'
        };
        
        // Procesar cada alerta
        alertas.forEach(alerta => {
            const lat = parseFloat(alerta.latitud) || 0;
            const lng = parseFloat(alerta.longitud) || 0;
            
            // Solo añadir si tiene coordenadas válidas
            if(lat !== 0 && lng !== 0) {
                const color = colores[alerta.tipo] || 'gray';
                const icono = iconos[alerta.tipo] || 'fa-bell';
                
                // Crear marcador personalizado
                const iconoPersonalizado = L.divIcon({
                    className: 'marcador-alerta',
                    html: `
                        <div class="marcador-contenido" style="
                            background: white;
                            border-radius: 50%;
                            width: 30px;
                            height: 30px;
                            display: flex;
                            align-items: center;
                            justify-content: center;
                            border: 2px solid ${color};
                            box-shadow: 0 2px 5px rgba(0,0,0,0.3);
                            color: ${color};
                        ">
                            <i class="fas ${icono}"></i>
                        </div>
                    `,
                    iconSize: [30, 30],
                    iconAnchor: [15, 30],
                    popupAnchor: [0, -30]
                });
                
                // Crear marcador
                const marcador = L.marker([lat, lng], { icon: iconoPersonalizado })
                    .addTo(capaAlertas);
                
                // Contenido del popup
                const popupContent = `
                    <div class="popup-alerta" style="min-width: 250px; font-family: sans-serif;">
                        <h4 style="margin: 0 0 10px 0; color: #2c3e50; border-bottom: 2px solid #3498db; padding-bottom: 8px;">
                            <i class="fas ${icono}"></i> ${alerta.titulo || 'Alerta'}
                        </h4>
                        <p><strong>Tipo:</strong> ${alerta.tipo}</p>
                        <p><strong>Prioridad:</strong> ${alerta.prioridad || 'media'}</p>
                        <p><strong>Ubicación:</strong> ${alerta.ubicacion || 'No especificada'}</p>
                        <p><strong>Fecha:</strong> ${alerta.fecha_formateada || alerta.fecha_creacion}</p>
                        <p class="descripcion" style="margin-top: 10px; padding: 10px; background: #f8f9fa; border-radius: 5px; border-left: 3px solid #3498db;">
                            ${alerta.descripcion || 'Sin descripción'}
                        </p>
                    </div>
                `;
                
                marcador.bindPopup(popupContent);
                marcadores.push(marcador);
            }
        });
        
        // Ajustar vista para mostrar todos los marcadores
        if(marcadores.length > 0) {
            const grupo = L.featureGroup(marcadores);
            mapa.fitBounds(grupo.getBounds().pad(0.1));
        }
        
        console.log(` ${marcadores.length} marcadores añadidos al mapa`);
        
    } catch(error) {
        console.error('Error cargando alertas en mapa:', error);
        cargarAlertasEjemploEnMapa();
    }
}



/**
 * Añade una nueva alerta al mapa
 */
function añadirAlertaAlMapa(alerta) {
    if(!mapa || !capaAlertas) return;
    
    const lat = parseFloat(alerta.latitud) || 0;
    const lng = parseFloat(alerta.longitud) || 0;
    
    if(lat === 0 || lng === 0) return;
    
    // Ícono según tipo
    const iconos = {
        'panico': L.divIcon({
            className: 'marcador-panico',
            html: '<i class="fas fa-exclamation-triangle fa-2x" style="color:red;"></i>',
            iconSize: [40, 40],
            iconAnchor: [20, 40]
        }),
        'default': L.divIcon({
            className: 'marcador-nuevo',
            html: '<i class="fas fa-map-marker-alt fa-2x" style="color:#e74c3c;"></i>',
            iconSize: [30, 30],
            iconAnchor: [15, 30]
        })
    };
    
    const icono = iconos[alerta.tipo] || iconos.default;
    
    const marcador = L.marker([lat, lng], { icon: icono })
        .addTo(capaAlertas)
        .bindPopup(`
            <b>${alerta.titulo}</b><br>
            <small>Tipo: ${alerta.tipo}</small><br>
            <small>Prioridad: ${alerta.prioridad}</small>
        `)
        .openPopup();
    
    marcadores.push(marcador);
    
    // Centrar en la nueva alerta
    mapa.setView([lat, lng], 16);
}

/**
 * Muestra mensaje de error en el mapa
 */
function mostrarMapaEjemplo(contenedor) {
    contenedor.innerHTML = `
        <div class="error-mapa" style="
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            height: 100%;
            padding: 30px;
            text-align: center;
            background: linear-gradient(135deg, #ffeded, #fff5f5);
            border-radius: 12px;
        ">
            <i class="fas fa-exclamation-triangle fa-3x" style="color:#e74c3c; margin-bottom:15px;"></i>
            <h3 style="color:#c0392b;">Mapa no disponible</h3>
            <p style="color:#666; margin-bottom:20px;">Se están cargando datos de ejemplo...</p>
            <div id="mapa-estatico" style="width:100%; height:300px; background:#f8f9fa; border-radius:8px; overflow:hidden;">
                <div style="width:100%; height:100%; background:url('https://maps.googleapis.com/maps/api/staticmap?center=-12.0464,-77.0428&zoom=12&size=600x300&scale=2&key=AIzaSyCExample') center/cover no-repeat;">
                </div>
            </div>
            <button onclick="inicializarMapa()" class="btn btn-small" style="
                background: #3498db;
                color: white;
                border: none;
                padding: 10px 20px;
                border-radius: 5px;
                margin-top: 15px;
                cursor: pointer;
            ">
                <i class="fas fa-redo"></i> Reintentar mapa
            </button>
        </div>
    `;
}

// ============================================
// INICIALIZACIÓN
// ============================================

// Inicializar cuando el DOM esté listo
document.addEventListener('DOMContentLoaded', function() {
    // Esperar un poco para que Leaflet se cargue
    setTimeout(inicializarMapa, 1000);
});

// Exportar funciones globales
window.inicializarMapa = inicializarMapa;
window.centrarMapaEn = centrarMapaEn;
window.añadirAlertaAlMapa = añadirAlertaAlMapa;
window.cargarAlertasEnMapa = cargarAlertasEnMapa;

console.log(' map.js cargado ');