// ============================================
// SISTEMA DE ALERTAS VES - APP.JS PRINCIPAL
// VERSIÓN SIMPLIFICADA - SOLO NOTIFICACIONES LOCALES
// ============================================

// Configuración global
window.API_BASE_URL = 'https://martin.monteroampsoluciones.com/backend';
window.currentLocation = null;
window.API_CONECTADO = false;
window.IS_PANIC_BUTTON_ACTIVE = false;

// ============================================
// FUNCIONES DE UTILIDAD
// ============================================

/**
 * Obtiene la URL base de la API
 */
function getApiBaseUrl() {
    return window.API_BASE_URL;
}

/**
 * Muestra una notificación en pantalla
 */
function mostrarNotificacion(tipo, mensaje, duracion = 5000) {
    try {
        let container = document.getElementById('notificaciones-container');
        if (!container) {
            container = document.createElement('div');
            container.id = 'notificaciones-container';
            container.className = 'notificaciones-container';
            container.style.cssText = `
                position: fixed;
                top: 20px;
                right: 20px;
                z-index: 9999;
                display: flex;
                flex-direction: column;
                gap: 10px;
                max-width: 350px;
            `;
            document.body.appendChild(container);
        }
        
        // Colores según tipo
        const colores = {
            'success': '#28a745',
            'error': '#dc3545',
            'warning': '#ffc107',
            'info': '#17a2b8',
            'panico': '#dc3545'
        };
        
        // Iconos según tipo
        const iconos = {
            'success': 'fa-check-circle',
            'error': 'fa-exclamation-circle',
            'warning': 'fa-exclamation-triangle',
            'info': 'fa-info-circle',
            'panico': 'fa-exclamation-triangle'
        };
        
        const notificacion = document.createElement('div');
        notificacion.className = `notificacion ${tipo}`;
        notificacion.style.cssText = `
            background: white;
            border-radius: 10px;
            padding: 15px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.2);
            display: flex;
            align-items: center;
            gap: 15px;
            border-left: 5px solid ${colores[tipo] || '#17a2b8'};
            animation: slideInRight 0.3s ease-out;
            max-width: 350px;
        `;
        
        notificacion.innerHTML = `
            <div style="
                font-size: 24px;
                width: 40px;
                height: 40px;
                border-radius: 50%;
                display: flex;
                align-items: center;
                justify-content: center;
                background: ${colores[tipo] || '#17a2b8'};
                color: white;
            ">
                <i class="fas ${iconos[tipo] || 'fa-info-circle'}"></i>
            </div>
            <div style="flex-grow: 1;">
                <div style="color: #333; font-weight: 500;">${mensaje}</div>
            </div>
            <button onclick="this.parentElement.remove()" style="
                background: none;
                border: none;
                color: #999;
                font-size: 18px;
                cursor: pointer;
                padding: 0;
                margin: 0;
                line-height: 1;
            ">
                <i class="fas fa-times"></i>
            </button>
        `;
        
        container.appendChild(notificacion);
        
        // Auto-eliminar
        setTimeout(() => {
            if (notificacion.parentNode) {
                notificacion.style.animation = 'fadeOut 0.5s ease-out forwards';
                setTimeout(() => notificacion.remove(), 500);
            }
        }, duracion);
        
        // Agregar estilos de animación si no existen
        if (!document.querySelector('#estilos-notificaciones')) {
            const estilos = document.createElement('style');
            estilos.id = 'estilos-notificaciones';
            estilos.textContent = `
                @keyframes slideInRight {
                    from { transform: translateX(100%); opacity: 0; }
                    to { transform: translateX(0); opacity: 1; }
                }
                @keyframes fadeOut {
                    from { transform: translateX(0); opacity: 1; }
                    to { transform: translateX(100%); opacity: 0; }
                }
                @keyframes pulse {
                    0% { box-shadow: 0 0 0 0 rgba(220, 53, 69, 0.7); }
                    70% { box-shadow: 0 0 0 10px rgba(220, 53, 69, 0); }
                    100% { box-shadow: 0 0 0 0 rgba(220, 53, 69, 0); }
                }
            `;
            document.head.appendChild(estilos);
        }
        
    } catch (error) {
        console.error('Error mostrando notificación:', error);
    }
}

/**
 * Cierra la notificación de emergencia
 */
function cerrarNotificacionEmergencia(button) {
    const notificacion = button.closest('.notificacion-emergencia');
    if (notificacion) {
        notificacion.style.display = 'none';
    }
}

/**
 * Muestra notificación de emergencia en la parte superior
 */
function mostrarNotificacionEmergencia(mensaje = '¡Alerta de pánico activada!') {
    const emergencia = document.getElementById('notificacion-emergencia');
    if (emergencia) {
        const mensajeElem = emergencia.querySelector('.mensaje p');
        if (mensajeElem) {
            mensajeElem.textContent = mensaje;
        }
        emergencia.style.display = 'block';
        
        // Auto-ocultar después de 15 segundos
        setTimeout(() => {
            emergencia.style.display = 'none';
        }, 15000);
    }
}

/**
 * Obtiene la ubicación actual del usuario
 */
async function obtenerUbicacionActual() {
    return new Promise((resolve, reject) => {
        if (!navigator.geolocation) {
            reject('Geolocalización no soportada');
            return;
        }
        
        navigator.geolocation.getCurrentPosition(
            (position) => {
                resolve({
                    lat: position.coords.latitude,
                    lng: position.coords.longitude,
                    accuracy: position.coords.accuracy
                });
            },
            (error) => {
                let mensaje = '';
                switch(error.code) {
                    case error.PERMISSION_DENIED:
                        mensaje = 'Permiso de ubicación denegado';
                        break;
                    case error.POSITION_UNAVAILABLE:
                        mensaje = 'Ubicación no disponible';
                        break;
                    case error.TIMEOUT:
                        mensaje = 'Tiempo de espera agotado';
                        break;
                    default:
                        mensaje = 'Error obteniendo ubicación';
                }
                reject(mensaje);
            },
            {
                enableHighAccuracy: true,
                timeout: 10000,
                maximumAge: 0
            }
        );
    });
}

// ============================================
// BOTÓN DE PÁNICO - SOLO NOTIFICACIÓN LOCAL
// ============================================

/**
 * Activa el botón de pánico - SOLO NOTIFICACIÓN PARA EL USUARIO ACTUAL
 */
async function activarBotonPanico() {
    // VERIFICACIÓN CRÍTICA: Evitar múltiples activaciones
    if (window.IS_PANIC_BUTTON_ACTIVE) {
        console.log(' Botón de pánico ya está activo, ignorando clic...');
        return;
    }
    
    const btnPanico = document.getElementById('btnPanico');
    if (!btnPanico) return;
    
    const originalHTML = btnPanico.innerHTML;
    
    try {
        // MARCADOR DE ESTADO: Prevenir nuevas activaciones
        window.IS_PANIC_BUTTON_ACTIVE = true;
        
        // Deshabilitar botón
        btnPanico.disabled = true;
        btnPanico.style.opacity = '0.7';
        btnPanico.innerHTML = '<i class="fas fa-spinner fa-spin"></i> ACTIVANDO...';
        
        // Notificación local de inicio
        mostrarNotificacion('warning', 'Activando botón de pánico...', 3000);
        
        console.log(' Iniciando activación de botón de pánico...');
        
        // Obtener ubicación
        let ubicacion;
        try {
            ubicacion = await obtenerUbicacionActual();
            console.log(' Ubicación obtenida:', ubicacion);
            mostrarNotificacion('success', 'Ubicación obtenida', 2000);
        } catch (error) {
            console.warn(' No se pudo obtener ubicación:', error);
            ubicacion = {
                lat: -12.0464,
                lng: -77.0428,
                accuracy: 0
            };
            mostrarNotificacion('info', 'Usando ubicación predeterminada', 2000);
        }
        
        // Preparar datos de alerta (SOLO se guarda en BD, no se comparte)
        const datosAlerta = {
            tipo: 'panico',
            titulo: 'BOTÓN DE PÁNICO ACTIVADO',
            descripcion: 'Usuario ha activado el botón de pánico. Se requiere asistencia inmediata.',
            ubicacion: `Lat: ${ubicacion.lat.toFixed(6)}, Lng: ${ubicacion.lng.toFixed(6)}`,
            latitud: ubicacion.lat,
            longitud: ubicacion.lng,
            prioridad: 'alta'
        };
        
        console.log(' Enviando alerta de pánico a la API...', datosAlerta);
        
        // Enviar a la API con timeout
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), 15000);
        
        const respuesta = await fetch(`${getApiBaseUrl()}/api.php?url=alertas`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json'
            },
            body: JSON.stringify(datosAlerta),
            signal: controller.signal
        });
        
        clearTimeout(timeoutId);
        
        console.log('Respuesta API recibida:', respuesta.status);
        
        if (!respuesta.ok) {
            throw new Error(`Error ${respuesta.status} al enviar alerta`);
        }
        
        const resultado = await respuesta.json();
        console.log('Resultado API:', resultado);
        
        if (resultado.success) {
            // NOTIFICACIÓN LOCAL SOLAMENTE - NO se envía a otros usuarios
            mostrarNotificacionEmergencia('¡Has activado el botón de pánico! La ayuda está en camino.');
            
            // Notificación toast de éxito
            mostrarNotificacion('success', 'Alerta de pánico enviada correctamente!', 5000);
            
            // Actualizar lista de alertas después de un delay
            if (typeof cargarAlertasRecientes === 'function') {
                setTimeout(() => {
                    cargarAlertasRecientes();
                    console.log('Lista de alertas actualizada');
                }, 2000);
            }
            
            console.log('Alerta de pánico creada exitosamente (solo para este usuario)');
            
        } else {
            throw new Error(resultado.message || 'Error al crear alerta');
        }
        
    } catch (error) {
        console.error('Error en botón de pánico:', error);
        
        if (error.name === 'AbortError') {
        } else {
            mostrarNotificacion('error', ' Error: ' + error.message, 5000);
        }
        
    } finally {
        // IMPORTANTE: Restaurar estado después de 5 segundos
        setTimeout(() => {
            const btnPanico = document.getElementById('btnPanico');
            if (btnPanico) {
                btnPanico.disabled = false;
                btnPanico.style.opacity = '1';
                btnPanico.innerHTML = '<i class="fas fa-exclamation-triangle"></i> BOTÓN DE PÁNICO';
            }
            
            // Liberar el bloqueo
            window.IS_PANIC_BUTTON_ACTIVE = false;
            console.log(' Botón de pánico listo para uso nuevamente');
        }, 5000);
    }
}

// ============================================
// FORMULARIO DE REPORTE - SOLO NOTIFICACIÓN LOCAL
// ============================================

/**
 * Maneja el envío del formulario de reporte
 */
async function manejarEnvioFormulario(event) {
    event.preventDefault();
    
    const formulario = event.target;
    const botonEnviar = formulario.querySelector('button[type="submit"]');
    if (!botonEnviar) return;
    
    const originalHTML = botonEnviar.innerHTML;
    
    try {
        // Deshabilitar botón
        botonEnviar.disabled = true;
        botonEnviar.innerHTML = '<i class="fas fa-spinner fa-spin"></i> ENVIANDO...';
        
        // Obtener datos del formulario
        const formData = new FormData(formulario);
        const datos = Object.fromEntries(formData.entries());
        
        // Validar campos requeridos
        if (!datos.tipo || !datos.titulo || !datos.descripcion) {
            throw new Error('Completa todos los campos requeridos');
        }
        
        // Obtener ubicación si no está definida
        if (!window.currentLocation) {
            try {
                window.currentLocation = await obtenerUbicacionActual();
                datos.latitud = window.currentLocation.lat;
                datos.longitud = window.currentLocation.lng;
                if (!datos.ubicacion) {
                    datos.ubicacion = `Lat: ${window.currentLocation.lat.toFixed(6)}, Lng: ${window.currentLocation.lng.toFixed(6)}`;
                }
            } catch (error) {
                console.warn('No se pudo obtener ubicación:', error);
            }
        } else {
            datos.latitud = window.currentLocation.lat;
            datos.longitud = window.currentLocation.lng;
        }
        
        // Validar ubicación
        if (!datos.ubicacion) {
            datos.ubicacion = 'Ubicación no especificada';
        }
        
        console.log(' Enviando reporte:', datos);
        
        // Enviar a la API
        const respuesta = await fetch(`${getApiBaseUrl()}/api.php?url=alertas`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json'
            },
            body: JSON.stringify(datos)
        });
        
        console.log(' Respuesta API:', respuesta.status);
        
        if (!respuesta.ok) {
            throw new Error(`Error ${respuesta.status} al enviar reporte`);
        }
        
        const resultado = await respuesta.json();
        
        if (resultado.success) {
            // NOTIFICACIÓN LOCAL SOLAMENTE
            mostrarNotificacion('success', '¡Reporte enviado exitosamente!', 5000);
            
            // Limpiar formulario
            formulario.reset();
            
            // Actualizar lista de alertas
            if (typeof cargarAlertasRecientes === 'function') {
                setTimeout(() => cargarAlertasRecientes(), 1000);
            }
            
            console.log(' Reporte creado:', resultado);
            
        } else {
            throw new Error(resultado.message || 'Error al enviar reporte');
        }
        
    } catch (error) {
        console.error(' Error enviando formulario:', error);
        mostrarNotificacion('error', 'Error: ' + error.message, 5000);
        
    } finally {
        // Restaurar botón después de 2 segundos
        setTimeout(() => {
            botonEnviar.disabled = false;
            botonEnviar.innerHTML = originalHTML;
        }, 2000);
    }
}

/**
 * Configura el botón de obtener ubicación
 */
function configurarBotonUbicacion() {
    const btnUbicacion = document.getElementById('btnUbicacion');
    const campoUbicacion = document.getElementById('incident-location');
    
    if (!btnUbicacion || !campoUbicacion) return;
    
    // Remover event listeners anteriores para evitar duplicados
    const nuevoBtn = btnUbicacion.cloneNode(true);
    btnUbicacion.parentNode.replaceChild(nuevoBtn, btnUbicacion);
    
    nuevoBtn.addEventListener('click', async () => {
        const originalHTML = nuevoBtn.innerHTML;
        
        try {
            nuevoBtn.disabled = true;
            nuevoBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> OBTENIENDO...';
            
            const ubicacion = await obtenerUbicacionActual();
            
            campoUbicacion.value = `Lat: ${ubicacion.lat.toFixed(6)}, Lng: ${ubicacion.lng.toFixed(6)}`;
            campoUbicacion.style.color = '#28a745';
            campoUbicacion.style.fontWeight = '500';
            
            window.currentLocation = ubicacion;
            
            mostrarNotificacion('success', 'Ubicación obtenida correctamente', 3000);
            
        } catch (error) {
            console.error('Error obteniendo ubicación:', error);
            
            campoUbicacion.value = 'Error al obtener ubicación';
            campoUbicacion.style.color = '#dc3545';
            
            mostrarNotificacion('error', `No se pudo obtener ubicación: ${error}`, 4000);
            
        } finally {
            setTimeout(() => {
                nuevoBtn.disabled = false;
                nuevoBtn.innerHTML = originalHTML;
            }, 1000);
        }
    });
}

// ============================================
// CONEXIÓN A API
// ============================================

/**
 * Verifica el estado de la API
 */
async function verificarConexionAPI() {
    try {
        console.log(' Verificando conexión a API...');
        
        const respuesta = await fetch(`${getApiBaseUrl()}/api.php?url=alertas`, {
            method: 'GET',
            headers: { 'Accept': 'application/json' }
        });
        
        if (respuesta.ok) {
            try {
                const data = await respuesta.json();
                window.API_CONECTADO = true;
                console.log(' API conectada correctamente');
                return true;
            } catch (jsonError) {
                console.error(' Error parseando JSON:', jsonError);
                mostrarNotificacion('error', 'Error en la respuesta del servidor', 5000);
                return false;
            }
        } else {
            console.error('API respondió con error:', respuesta.status);
            mostrarNotificacion('error', `Error ${respuesta.status} del servidor`, 5000);
            return false;
        }
        
    } catch (error) {
        console.error(' Error de conexión a API:', error.message);
        mostrarNotificacion('error', 'No se puede conectar al servidor', 5000);
        return false;
    }
}

// ============================================
// INICIALIZACIÓN DE LA APLICACIÓN
// ============================================

/**
 * Inicializa la aplicación completa
 */
function inicializarApp() {

    try {
        // 1. Configurar botón de pánico
        const btnPanico = document.getElementById('btnPanico');
        if (btnPanico) {
            // Remover cualquier listener anterior para evitar duplicados
            const nuevoBtn = btnPanico.cloneNode(true);
            btnPanico.parentNode.replaceChild(nuevoBtn, btnPanico);
            
            // Agregar un solo listener
            nuevoBtn.addEventListener('click', function(e) {
                e.preventDefault();
                e.stopPropagation();
                activarBotonPanico();
            });
            

        }
        
        // 2. Configurar formulario
        const formularioAlerta = document.getElementById('alertaForm');
        if (formularioAlerta) {
            formularioAlerta.addEventListener('submit', manejarEnvioFormulario);
        }
        
        // 3. Configurar botón de ubicación
        configurarBotonUbicacion();

        // 4. Verificar API
        setTimeout(() => {
            verificarConexionAPI();
        }, 2000);
        
        // 5. Cargar alertas iniciales
        if (typeof cargarAlertasRecientes === 'function') {
            setTimeout(() => {
                cargarAlertasRecientes();
            }, 1500);
        }
        
        // 6. Notificación de bienvenida
        setTimeout(() => {
            mostrarNotificacion('info', 
                'Sistema de Alertas VES listo. ¡Tu seguridad es nuestra prioridad!', 
                5000);
        }, 3000);
        
        console.log(' Aplicación inicializada correctamente');
        
    } catch (error) {
        console.error(' Error inicializando aplicación:', error);
        mostrarNotificacion('error', 'Error inicializando la aplicación', 5000);
    }
}

// ============================================
// MANEJO DE EVENTOS
// ============================================

// Inicializar cuando el DOM esté listo
document.addEventListener('DOMContentLoaded', () => {
    console.log(' DOM cargado, iniciando aplicación...');
    setTimeout(inicializarApp, 500);
});

// Manejar errores no capturados
window.addEventListener('error', (event) => {
    console.error(' Error no capturado:', event.error);
});

// ============================================
// EXPORTACIÓN DE FUNCIONES GLOBALES
// ============================================

window.mostrarNotificacion = mostrarNotificacion;
window.mostrarNotificacionEmergencia = mostrarNotificacionEmergencia;
window.cerrarNotificacionEmergencia = cerrarNotificacionEmergencia;
window.activarBotonPanico = activarBotonPanico;
window.obtenerUbicacionActual = obtenerUbicacionActual;
window.manejarEnvioFormulario = manejarEnvioFormulario;
window.verificarConexionAPI = verificarConexionAPI;
window.getApiBaseUrl = getApiBaseUrl;

console.log(' App.js cargado');