<?php
// MICROSERVICIO DE ALERTAS - VERSIÓN SIMPLIFICADA

// ============================================
// CONFIGURACIÓN DE ZONA HORARIA - PERÚ (LIMA)
// ============================================

// Forzar zona horaria de Perú independientemente del servidor
if (function_exists('date_default_timezone_set')) {
    date_default_timezone_set('America/Lima');
}

// También ajustar para la sesión PHP
ini_set('date.timezone', 'America/Lima');

// 2. Headers para JSON y CORS
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With');

// 3. Manejar CORS preflight
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// 4. Función para conexión a BD
function conectarBD() {
    $host = 'localhost';
    $dbname = 'monteroa_alerta_ves';
    $username = 'monteroa_alerta_ves';
    $password = 'AlertaVES2025!';
    
    try {
        $conn = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $username, $password);
        $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $conn->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
        return $conn;
    } catch (PDOException $e) {
        error_log(" ERROR BD en ms-alertas: " . $e->getMessage());
        return null;
    }
}

// 5. Determinar método HTTP
$method = $_SERVER['REQUEST_METHOD'];

// ============================================
// GET: OBTENER ALERTAS RECIENTES
// ============================================
if ($method === 'GET') {
    $conn = conectarBD();
    
    if (!$conn) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Error de conexión a la base de datos',
            'timestamp' => date('Y-m-d H:i:s')
        ]);
        exit();
    }
    
    try {
        // Consulta con valores por defecto para coordenadas nulas
        $sql = "SELECT id, tipo, titulo, descripcion, ubicacion, 
                       IFNULL(latitud, -12.0464) as latitud, 
                       IFNULL(longitud, -77.0428) as longitud,
                       prioridad, fecha_creacion
                FROM alertas 
                WHERE estado != 'cancelada' OR estado IS NULL
                ORDER BY fecha_creacion DESC 
                LIMIT 15";
        
        $stmt = $conn->query($sql);
        $alertas = $stmt->fetchAll();
        
        // Formatear respuesta
        foreach ($alertas as &$alerta) {
            $fecha = new DateTime($alerta['fecha_creacion']);
            $alerta['fecha_formateada'] = $fecha->format('d/m/Y H:i:s');
            $alerta['dia'] = $fecha->format('d/m/Y');
            $alerta['hora'] = $fecha->format('H:i');
            
            // Icono según tipo
            $iconos = [
                'panico' => 'fa-exclamation-triangle',
                'robo' => 'fa-gem',
                'accidente' => 'fa-car-crash',
                'incendio' => 'fa-fire',
                'violencia' => 'fa-user-injured',
                'otro' => 'fa-exclamation-circle'
            ];
            $alerta['icono'] = $iconos[$alerta['tipo']] ?? 'fa-bell';
        }
        
        echo json_encode([
            'success' => true,
            'count' => count($alertas),
            'data' => $alertas,
            'timestamp' => date('Y-m-d H:i:s')
        ], JSON_UNESCAPED_UNICODE);
        
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Error en la consulta a la base de datos',
            'error' => $e->getMessage(),
            'timestamp' => date('Y-m-d H:i:s')
        ]);
    }
    
    exit();
}

// ============================================
// POST: CREAR ALERTA
// ============================================
elseif ($method === 'POST') {
    $conn = conectarBD();
    
    if (!$conn) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Error de conexión a la base de datos'
        ]);
        exit();
    }
    
    // Obtener datos de entrada
    $contentType = $_SERVER['CONTENT_TYPE'] ?? '';
    $input = [];
    
    if (strpos($contentType, 'application/json') !== false) {
        $input = json_decode(file_get_contents('php://input'), true) ?: [];
    } else {
        $input = $_POST;
    }
    
    // Validar datos básicos
    if (empty($input)) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'No se recibieron datos en la solicitud'
        ]);
        exit();
    }
    
    try {
        // Procesar datos
        $tipo = $input['tipo'] ?? 'otro';
        $esPanico = ($tipo === 'panico');
        
        if ($esPanico) {
            $titulo = ' BOTÓN DE PÁNICO ACTIVADO';
            $descripcion = 'Usuario ha activado el botón de pánico. Se requiere asistencia inmediata.';
            $prioridad = 'alta';
            $latitud = $input['latitud'] ?? -12.0464;
            $longitud = $input['longitud'] ?? -77.0428;
            $ubicacion = $input['ubicacion'] ?? "Lat: $latitud, Lng: $longitud";
        } else {
            $titulo = htmlspecialchars($input['titulo'] ?? 'Alerta sin título', ENT_QUOTES, 'UTF-8');
            $descripcion = htmlspecialchars($input['descripcion'] ?? '', ENT_QUOTES, 'UTF-8');
            $prioridad = in_array($input['prioridad'] ?? 'media', ['baja', 'media', 'alta']) ? 
                         $input['prioridad'] : 'media';
            $ubicacion = htmlspecialchars($input['ubicacion'] ?? 'Ubicación no especificada', ENT_QUOTES, 'UTF-8');
            $latitud = isset($input['latitud']) ? floatval($input['latitud']) : null;
            $longitud = isset($input['longitud']) ? floatval($input['longitud']) : null;
        }
        
        // Insertar en BD
        $sql = "INSERT INTO alertas (tipo, titulo, descripcion, ubicacion, latitud, longitud, prioridad, fecha_creacion) 
                VALUES (:tipo, :titulo, :descripcion, :ubicacion, :latitud, :longitud, :prioridad, NOW())";
        
        $stmt = $conn->prepare($sql);
        $stmt->execute([
            ':tipo' => $tipo,
            ':titulo' => $titulo,
            ':descripcion' => $descripcion,
            ':ubicacion' => $ubicacion,
            ':latitud' => $latitud,
            ':longitud' => $longitud,
            ':prioridad' => $prioridad
        ]);
        
        $idAlerta = $conn->lastInsertId();
        
        // Obtener alerta creada para respuesta
        $stmt = $conn->prepare("SELECT * FROM alertas WHERE id = ?");
        $stmt->execute([$idAlerta]);
        $alertaCreada = $stmt->fetch();
        
        // Formatear fecha
        $fecha = new DateTime($alertaCreada['fecha_creacion']);
        $fechaFormateada = $fecha->format('d/m/Y H:i:s');
        
        // Respuesta exitosa
        echo json_encode([
            'success' => true,
            'message' => $esPanico ? '¡Alerta de pánico enviada exitosamente!' : 'Alerta creada exitosamente',
            'id_alerta' => $idAlerta,
            'tipo' => $tipo,
            'fecha_creacion' => $fechaFormateada,
            'alerta' => $alertaCreada
        ], JSON_UNESCAPED_UNICODE);
        
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Error al crear la alerta',
            'error' => $e->getMessage()
        ]);
    }
    
    exit();
}

// ============================================
// MÉTODO NO SOPORTADO
// ============================================
else {
    http_response_code(405);
    echo json_encode([
        'success' => false,

        'allowed_methods' => ['GET', 'POST']
    ]);
}
?>