<?php
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../models/Notificacion.php';

class NotificacionController {
    private $notificacion;
    
    public function __construct() {
        try {
            $this->notificacion = new Notificacion();
        } catch (Exception $e) {
            error_log("Error en NotificacionController: " . $e->getMessage());
            throw $e;
        }
    }
    
    public function createNotificacion($data) {
        try {
            // Validar que los datos necesarios existan
            if (!isset($data->mensaje) || !isset($data->destinatario)) {
                return array("success" => false, "message" => "Datos incompletos: mensaje y destinatario son obligatorios");
            }
            
            $this->notificacion->tipo = isset($data->tipo) ? $data->tipo : 'email';
            $this->notificacion->destinatario = is_array($data->destinatario) ? implode(',', $data->destinatario) : $data->destinatario;
            $this->notificacion->mensaje = $data->mensaje;
            
            if ($this->notificacion->create()) {
                return array("success" => true, "message" => "Notificación registrada correctamente");
            } else {
                return array("success" => false, "message" => "No se pudo registrar la notificación en la base de datos");
            }
        } catch(Exception $e) {
            error_log("Error en createNotificacion: " . $e->getMessage());
            return array("success" => false, "message" => "Error al crear notificación: " . $e->getMessage());
        }
    }
    
    public function getNotificaciones() {
        try {
            $stmt = $this->notificacion->read();
            $num = $stmt->rowCount();
            
            if ($num > 0) {
                $notificaciones_arr = array();
                $notificaciones_arr["success"] = true;
                $notificaciones_arr["notifications"] = array();
                
                while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                    $notificaciones_arr["notifications"][] = $row;
                }
                
                return $notificaciones_arr;
            } else {
                return array("success" => true, "notifications" => array());
            }
        } catch(Exception $e) {
            error_log("Error en getNotificaciones: " . $e->getMessage());
            return array("success" => false, "message" => "Error al obtener notificaciones: " . $e->getMessage());
        }
    }
    
    public function processPendingNotifications() {
        try {
            // Obtener notificaciones pendientes
            $query = "SELECT * FROM notificaciones WHERE estado = 'pendiente' LIMIT 10";
            $stmt = $this->notificacion->conn->prepare($query);
            $stmt->execute();
            
            $notifications = $stmt->fetchAll(PDO::FETCH_ASSOC);
            $processed = 0;
            
            foreach ($notifications as $notification) {
                // Procesar cada notificación según su tipo
                $result = $this->sendNotification($notification);
                
                if ($result['success']) {
                    // Marcar como enviada
                    $this->notificacion->markAsSent($notification['id']);
                    $processed++;
                }
            }
            
            return array(
                "success" => true, 
                "message" => "Procesadas $processed notificaciones de " . count($notifications)
            );
            
        } catch(Exception $e) {
            error_log("Error en processPendingNotifications: " . $e->getMessage());
            return array("success" => false, "message" => "Error al procesar notificaciones: " . $e->getMessage());
        }
    }
    
    private function sendNotification($notification) {
        try {
            switch ($notification['tipo']) {
                case 'email':
                    return $this->sendEmail($notification);
                case 'sms':
                    return $this->sendSMS($notification);
                case 'push':
                    return $this->sendPush($notification);
                default:
                    return array("success" => false, "message" => "Tipo de notificación no soportado");
            }
        } catch (Exception $e) {
            error_log("Error en sendNotification: " . $e->getMessage());
            return array("success" => false, "message" => "Error al enviar notificación: " . $e->getMessage());
        }
    }
    
    private function sendEmail($notification) {
        try {
            $to = $notification['destinatario'];
            $subject = 'Nueva Alerta - Alerta VES';
            $message = $notification['mensaje'];
            
            $headers = "MIME-Version: 1.0" . "\r\n";
            $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
            $headers .= 'From: Alerta VES <noreply@alertaves.com>' . "\r\n";
            
            $success = mail($to, $subject, $message, $headers);
            
            if ($success) {
                return array("success" => true, "message" => "Email enviado correctamente");
            } else {
                return array("success" => false, "message" => "Error al enviar email");
            }
        } catch (Exception $e) {
            return array("success" => false, "message" => "Excepción al enviar email: " . $e->getMessage());
        }
    }
    
    private function sendSMS($notification) {
        try {
            // Aquí implementarías el envío de SMS usando un servicio como Twilio
            // Por ahora, solo simulamos el envío
            error_log("SMS enviado a " . $notification['destinatario'] . ": " . $notification['mensaje']);
            
            return array("success" => true, "message" => "SMS enviado correctamente");
        } catch (Exception $e) {
            return array("success" => false, "message" => "Error al enviar SMS: " . $e->getMessage());
        }
    }
    
    private function sendPush($notification) {
        try {
            // Aquí implementarías el envío de notificaciones push usando Firebase Cloud Messaging
            // Por ahora, solo simulamos el envío
            error_log("Push notification enviada a " . $notification['destinatario'] . ": " . $notification['mensaje']);
            
            return array("success" => true, "message" => "Push notification enviada correctamente");
        } catch (Exception $e) {
            return array("success" => false, "message" => "Error al enviar push notification: " . $e->getMessage());
        }
    }
}
?>